import datetime
import os 
import sys 
import csv
import re
import time
import glob
import pandas as pd
import matplotlib.pyplot as plt
import logging
from natsort import natsorted

sys.path.append('/path to methodscript resources => MethodSCRIPT_Examples/MethodSCRIPTExample_Python/MethodSCRIPTExample_Python')

import palmsens.instrument
import palmsens.mscript
import palmsens.serial

DEVICE_PORT = None
MSCRIPT_FILE_PATH = '/path to methodscript'
OUTPUT_PATH = os.environ.get("EXPERIMENT_RESULTS_FOLDER", "/tmp/potentiostat_output")
os.makedirs(OUTPUT_PATH, exist_ok=True)

# Create Potentiostat subfolder
POTENTIOSTAT_FOLDER = os.path.join(OUTPUT_PATH, "Potentiostat")
os.makedirs(POTENTIOSTAT_FOLDER, exist_ok=True)

# Configure logging to file only (no console output) - save in Potentiostat folder
log_file = os.path.join(POTENTIOSTAT_FOLDER, "potentiostat_log.txt")
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(message)s",
    handlers=[logging.FileHandler(log_file, mode="w")]
)

TECHNIQUE_LABELS = {
    'CV':  {'xlabel': 'Applied Potential (V)', 'ylabel': 'Measured Current (A)'},
    'LSV': {'xlabel': 'Applied Potential (V)', 'ylabel': 'Measured Current (A)'},
    'CA':  {'xlabel': 'Time (s)',              'ylabel': 'Measured Current (A)'},
    'CP':  {'xlabel': 'Time (s)',              'ylabel': 'Measured Potential (V)'},
    'OCP': {'xlabel': 'Time (s)',              'ylabel': 'Measured Potential (V)'},
}

COLUMN_MAP = {
    'CV':  (0, 1),
    'LSV': (0, 1),
    'CA':  (0, 2),
    'CP':  (0, 2),
    'OCP': (0, 1),
}

def get_technique_list_from_methodscript(mscript_path):
    techniques = []
    with open(mscript_path, 'r') as f:
        for line in f:
            match = re.search(r'meas_loop_([a-z]+)', line, re.IGNORECASE)
            if match:
                technique = match.group(1).upper()
                techniques.append(technique)
    return techniques if techniques else ['CV']

def split_measurements_by_asterisk(lines):
    chunks = []
    current = []
    for line in lines:
        current.append(line)
        if line.strip() == '*':
            chunks.append(current)
            current = []
    if current:
        chunks.append(current)
    return chunks

def combine_and_plot_technique_csvs(technique):
    """Combine all CSV files for a given technique and create plots"""
    pattern = os.path.join(POTENTIOSTAT_FOLDER, f"measurement_{technique}_*.csv")
    csv_files = natsorted(glob.glob(pattern))  # Sort files to preserve order
    
    if not csv_files:
        logging.info(f"No CSV files found for technique {technique}")
        return
    
    dfs = [pd.read_csv(f) for f in csv_files]
    combined_df = pd.concat(dfs, ignore_index=True)
    
    combined_csv_name = datetime.datetime.now().strftime(f'measurements_{technique}_%Y%m%d-%H%M%S.csv')
    combined_csv_path = os.path.join(POTENTIOSTAT_FOLDER, combined_csv_name)
    combined_df.to_csv(combined_csv_path, index=False)
    logging.info(f"Saved combined CSV for {technique}: {combined_csv_path}")

    # Plot the first two columns
    plt.figure(figsize=(10, 6))
    plt.plot(combined_df.iloc[:, 0], combined_df.iloc[:, 1], label=technique)
    plt.xlabel(combined_df.columns[0])
    plt.ylabel(combined_df.columns[1])
    plt.title(f"Measurement {technique}")
    plt.legend()
    plt.grid(True)

    plt.gca().relim()
    plt.gca().autoscale_view()

    plot_path = os.path.join(POTENTIOSTAT_FOLDER, f"Measurement_{technique}.png")
    plt.savefig(plot_path, bbox_inches='tight')
    plt.close()
    logging.info(f"Saved combined plot for {technique} at: {plot_path}")
    plt.show()  # remove if you don't want to display

    # Remove individual CSVs
    for file in csv_files:
        try:
            os.remove(file)
            logging.info(f"Deleted individual CSV file: {file}")
        except Exception as e:
            logging.error(f"Failed to delete {file}: {e}")

def main():

    logging.info(f"Python version: {sys.version}")
    logging.info(f"Working dir: {os.getcwd()}")
    logging.info(f"Using MethodSCRIPT: {MSCRIPT_FILE_PATH}")
    logging.info(f"Potentiostat output folder: {POTENTIOSTAT_FOLDER}")
    
    port = DEVICE_PORT
    if port is None:
        port = palmsens.serial.auto_detect_port()

    with palmsens.serial.Serial(port, 1) as comm:
        device = palmsens.instrument.Instrument(comm)
        device.abort_and_sync()
        device_type = device.get_device_type()
        logging.info(f'Connected to {device_type}.')
        fw_version = device.get_firmware_version()
        serial_number = device.get_serial_number()
        logging.info(f"Firmware: {fw_version}, Serial: {serial_number}")
        
        # Mark the time as measurement start (sync point)
        # NOTE: measurement_start.txt stays in OUTPUT_PATH root, NOT in Potentiostat folder
        measurement_start = time.time()
        measurement_start_path = os.path.join(OUTPUT_PATH, "measurement_start.txt")
        with open(measurement_start_path, "w") as f:
            f.write(f"PotentiostatStart_wallclock={measurement_start}\n")
            f.flush()
            os.fsync(f.fileno())

        logging.info("Potentiostat wrote sync file, waiting for camera...")

        # Wait until camera appends CameraFirstFrame_wallclock
        while True:
            with open(measurement_start_path, "r") as f:
                lines = f.read()
            if "CameraFirstFrame_wallclock=" in lines:
                logging.info("Camera timestamp found, continuing measurement loop.")
                break
            time.sleep(0.01)        
        
        device.send_script(MSCRIPT_FILE_PATH)        
        # Read lines
        result_lines = device.readlines_until_end()

        # Collect raw output lines         
        # Log raw device output for full traceability
        logging.info("===== Raw Potentiostat Output Start =====")
        for line in result_lines:
            logging.info(line.strip())
        logging.info("===== Raw Potentiostat Output End =====")

    # Process parsed results
    technique_list = get_technique_list_from_methodscript(MSCRIPT_FILE_PATH)
    measurement_chunks = split_measurements_by_asterisk(result_lines)

    for idx, chunk in enumerate(measurement_chunks, start=1):
        technique = technique_list[idx-1] if idx-1 < len(technique_list) else technique_list[-1]

        # Map technique to correct axis labels and data columns
        labels = TECHNIQUE_LABELS.get(technique, {'xlabel': 'X', 'ylabel': 'Y'})
        x_col, y_col = COLUMN_MAP.get(technique, (0, 1))

        curves = palmsens.mscript.parse_result_lines(chunk)
        x_vals = palmsens.mscript.get_values_by_column(curves, x_col)
        y_vals = palmsens.mscript.get_values_by_column(curves, y_col)

        if len(x_vals) == 0 or len(y_vals) == 0:
            continue

        csv_file_name = datetime.datetime.now().strftime(
            f'measurement_{technique}_{idx}_%Y%m%d-%H%M%S.csv'
        )
        # Save CSV in Potentiostat folder
        csv_path = os.path.join(POTENTIOSTAT_FOLDER, csv_file_name)

        with open(csv_path, 'w', newline='') as csvfile:
            writer = csv.writer(csvfile)
            writer.writerow([labels['xlabel'], labels['ylabel']])
            writer.writerows(zip(x_vals, y_vals))
        logging.info(f"Saved raw CSV for {technique} #{idx}: {csv_path}")

    # Combine and plot all techniques after saving individual CSVs
    for technique in set(technique_list):
        combine_and_plot_technique_csvs(technique)

    logging.info("Measurement complete.")

if __name__ == '__main__':
    main()