# -*- coding: utf-8 -*-
"""
Automated ECL Analysis Script
Called by camera code with all parameters passed via command line
Analyzes both PNG and TIFF images using the same ROIs
"""

import os
import sys
import glob
import numpy as np
import pandas as pd
from PIL import Image
import tifffile 
import cv2
import matplotlib.pyplot as plt

# ============================================================================
# PARSE COMMAND LINE ARGUMENTS
# ============================================================================
if len(sys.argv) < 15:
    print("Usage: python ecl_analysis_auto.py <results_folder> <framerate> <cv_start_pot> " +
          "<cv_end_pot> <cv_n_cycles> <cv_scan_rate> <ca_start_time> <electrode_area> " +
          "<ca_applied_potential> <roi_shape> <num_rois> <fixed_radius> <fixed_width> <fixed_height>")
    sys.exit(1)

main_folder = sys.argv[1]
framerate = float(sys.argv[2])
cv_start_pot = float(sys.argv[3])
cv_end_pot = float(sys.argv[4])
cv_n_cycles = int(sys.argv[5])
cv_scan_rate = float(sys.argv[6])
ca_start_time = float(sys.argv[7])
electrode_area = float(sys.argv[8])
ca_applied_potential = float(sys.argv[9])
roi_shape = sys.argv[10]  # "circle" or "rectangle"
num_rois = int(sys.argv[11])
fixed_radius = int(sys.argv[12])
fixed_width = int(sys.argv[13])
fixed_height = int(sys.argv[14])

# Convert roi_shape to internal format
roi_choice = 'c' if roi_shape == "circle" else 'r'

print(f"\nAnalyzing results in: {main_folder}")
print(f"Frame rate: {framerate} fps")
print(f"Electrode area: {electrode_area} cm²")
print(f"ROI shape: {roi_shape}, Count: {num_rois}")

# ============================================================================
# FOLDER STRUCTURE DETECTION
# ============================================================================
if not os.path.isdir(main_folder):
    raise FileNotFoundError(f"Results folder not found: {main_folder}")

# Frames base folder
frames_base = os.path.join(main_folder, "Frames")
if not os.path.isdir(frames_base):
    raise FileNotFoundError(f"Frames folder not found in {main_folder}")

# PNG frames folder
png_frames_folder = os.path.join(frames_base, "png")
if not os.path.isdir(png_frames_folder):
    raise FileNotFoundError(f"png folder not found in {frames_base}")

# TIFF frames folder
tiff_frames_folder = os.path.join(frames_base, "raw_tiff")
if not os.path.isdir(tiff_frames_folder):
    raise FileNotFoundError(f"raw_tiff folder not found in {frames_base}")

# ROI frame folder
roi_frame_folder = os.path.join(frames_base, "roi_frame")
if not os.path.isdir(roi_frame_folder):
    raise FileNotFoundError(f"roi_frame folder not found in {frames_base}")

# Find the unique ROI frame file
roi_frame_files = glob.glob(os.path.join(roi_frame_folder, "frame_roi_*.png"))
if not roi_frame_files:
    raise FileNotFoundError(f"No ROI frame file found in {roi_frame_folder}")
roi_frame_path = roi_frame_files[0]
print(f"Found ROI frame: {os.path.basename(roi_frame_path)}")

# Find measurement CSV (*measurement*.csv)
csv_files = glob.glob(os.path.join(main_folder, "Potentiostat", "*measurement*.csv"))
if not csv_files:
    raise FileNotFoundError(f"No CSV file with 'measurement' in name found in {main_folder}")
ec_csv_path = csv_files[0]
print(f"Found measurement CSV: {os.path.basename(ec_csv_path)}")

# Background images (frame_sync)
png_bg_path = os.path.join(png_frames_folder, "frame_sync.png")
if not os.path.exists(png_bg_path):
    raise FileNotFoundError(f"PNG background frame_sync not found in {png_frames_folder}")

tiff_bg_candidates = glob.glob(os.path.join(tiff_frames_folder, "frame_sync.*"))
if not tiff_bg_candidates:
    raise FileNotFoundError(f"TIFF background frame_sync not found in {tiff_frames_folder}")
tiff_bg_path = tiff_bg_candidates[0]

# Get frame files (exclude frame_sync)
png_frame_files = sorted([
    f for f in glob.glob(os.path.join(png_frames_folder, "frame_*.png"))
    if os.path.abspath(f) != os.path.abspath(png_bg_path)
])

tiff_frame_files = sorted([
    f for f in glob.glob(os.path.join(tiff_frames_folder, "frame_*.tiff"))
    if os.path.abspath(f) != os.path.abspath(tiff_bg_path)
])

print(f"Found {len(png_frame_files)} PNG frames and {len(tiff_frame_files)} TIFF frames to analyze")

# ============================================================================
# HELPER FUNCTIONS
# ============================================================================
def load_image(path):
    ext = path.lower().split('.')[-1]
    if ext in ['tif', 'tiff']:
        return tifffile.imread(path)
    else:
        return np.array(Image.open(path))

def extract_red_channel(img):
    if img.ndim == 3 and img.shape[2] >= 3:
        return img[:, :, 0]
    else:
        return img

def auto_brightness_contrast(img, saturated=0.35):
    flat = img.flatten()
    bins = 256 if img.dtype == np.uint8 else 65536 if img.dtype == np.uint16 else 1024
    hist, bin_edges = np.histogram(flat, bins=bins, range=(flat.min(), flat.max()))
    total_pixels = flat.size
    cutoff = int(total_pixels * saturated / 100)
    cumulative = np.cumsum(hist)
    min_val = bin_edges[np.searchsorted(cumulative, cutoff)]
    max_val = bin_edges[np.searchsorted(cumulative, total_pixels - cutoff)]
    if min_val == max_val: min_val, max_val = flat.min(), flat.max()
    return np.clip((img - min_val) * 255.0 / (max_val - min_val), 0, 255).astype(np.uint8)

def select_circle_roi(img, radius):
    roi = {'center': (0, 0)}
    
    def mouse_cb(event, x, y, flags, param):
        if event == cv2.EVENT_LBUTTONDOWN:
            roi['center'] = (x, y)
    
    window_name = f"Select ROI Center (circle r={radius}) - Click center then press Enter/Space"
    cv2.namedWindow(window_name) #    cv2.namedWindow(window_name, cv2.WINDOW_NORMAL)
    cv2.setMouseCallback(window_name, mouse_cb)
    
    while True:
        temp = img.copy()
        cx, cy = roi['center']
        if cx != 0 or cy != 0:
            cv2.circle(temp, (cx, cy), radius, (255, 255, 255), 2)
        cv2.imshow(window_name, temp)
        key = cv2.waitKey(1) & 0xFF
        if key in [13, 32]: #Enter or space confirms
            break
        if key == 27:
            roi['center'] = (0, 0) #Esc cancels
            break
    
    cv2.destroyWindow(window_name)
    
    if roi['center'] == (0, 0):
        print("No roi selected.")
        return None
    return roi['center'][0], roi['center'][1], radius

def select_rectangle_roi(img, width, height):
    roi = {'center': (0, 0)}
    
    def mouse_cb(event, x, y, flags, param):
        if event == cv2.EVENT_LBUTTONDOWN:
            roi['center'] = (x, y)
    
    window_name = f"Select ROI Center (rect {width}x{height}) - Click center then press Enter/Space"
    cv2.namedWindow(window_name) #cv2.namedWindow(window_name, cv2.WINDOW_NORMAL)
    cv2.setMouseCallback(window_name, mouse_cb)
    
    while True:
        temp = img.copy()
        cx, cy = roi['center']
        if cx != 0 or cy != 0:
            x1 = max(cx - width // 2, 0)
            y1 = max(cy - height // 2, 0)
            x2 = min(cx + width // 2, img.shape[1] - 1)
            y2 = min(cy + height // 2, img.shape[0] - 1)
            cv2.rectangle(temp, (x1, y1), (x2, y2), (255, 255, 255), 2)
        cv2.imshow(window_name, temp)
        key = cv2.waitKey(1) & 0xFF
        if key in [13, 32]: #Enter or space confirms
            break
        if key == 27: #Esc cancels
            roi['center'] = (0, 0)
            break
    
    cv2.destroyWindow(window_name)
    
    if roi['center'] == (0, 0):
        print("No roi selected.")
        return None
    cx, cy = roi['center']
    x = max(cx - width // 2, 0)
    y = max(cy - height // 2, 0)
    return x, y, width, height

def sum_roi(img, shape, params):
    if shape == "rect":
        x, y, w, h = params
        roi = img[y:y + h, x:x + w]
        return int(np.sum(roi)), roi.size
    elif shape == "circle":
        cx, cy, r = params
        y_min, y_max = max(0, cy - r), min(img.shape[0], cy + r)
        x_min, x_max = max(0, cx - r), min(img.shape[1], cx + r)
        roi = img[y_min:y_max, x_min:x_max]
        yy, xx = np.ogrid[:roi.shape[0], :roi.shape[1]]
        mask = (yy - (cy - y_min)) ** 2 + (xx - (cx - x_min)) ** 2 <= r ** 2
        pixels = roi[mask]
        return int(np.sum(pixels)), pixels.size
    return 0, 0

def select_multiple_rois_sequential(img_display, roi_choice, num_rois, radius, width, height):
    rois = []
    working_img = img_display.copy()
    for i in range(num_rois):
        print(f"Select ROI {i + 1} of {num_rois}. Click center then press Enter/Space.")
        if roi_choice == 'c':
            sel = select_circle_roi(working_img, radius)
            if sel is None:
                print("Selection cancelled. Stopping ROI selection.")
                break
            rois.append(sel)
            cx, cy, r = sel
            cv2.circle(working_img, (cx, cy), r, (255, 255, 255), 2)
        else:
            sel = select_rectangle_roi(working_img, width, height)
            if sel is None:
                print("Selection cancelled. Stopping ROI selection.")
                break
            rois.append(sel)
            x, y, w, h = sel
            cv2.rectangle(working_img, (x, y), (x + w, y + h), (255, 255, 255), 2)
    return rois

def draw_rois_on_image(img, roi_list, roi_shape='circle'):
    overlay_img = cv2.cvtColor(img.copy(), cv2.COLOR_GRAY2BGR) if img.ndim == 2 else img.copy()
    for idx, rp in enumerate(roi_list):
        color = (0, 255, 0)
        if roi_shape == 'circle':
            cx, cy, r = rp
            cv2.circle(overlay_img, (cx, cy), r, color, 2)
            cv2.putText(overlay_img, f"ROI{idx + 1}", (cx - r, cy - r - 5),
                        cv2.FONT_HERSHEY_SIMPLEX, 0.5, color, 1)
        else:
            x, y, w, h = rp
            cv2.rectangle(overlay_img, (x, y), (x + w, y + h), color, 2)
            cv2.putText(overlay_img, f"ROI{idx + 1}", (x, y - 5),
                        cv2.FONT_HERSHEY_SIMPLEX, 0.5, color, 1)
    return overlay_img

def generate_frame_potentials(cv_start_pot, cv_end_pot, cv_scan_rate, cv_n_cycles, frame_count, framerate):
    sweep_time = abs(cv_end_pot - cv_start_pot) / cv_scan_rate
    cycle_time = sweep_time * 2
    t = (np.arange(frame_count) + 1) / framerate
    mod_t = np.mod(t, cycle_time)
    potentials = np.where(
        mod_t < sweep_time,
        cv_start_pot + (cv_end_pot - cv_start_pot) * (mod_t / sweep_time),
        cv_end_pot - (cv_end_pot - cv_start_pot) * ((mod_t - sweep_time) / sweep_time),
    )
    return potentials, t

def generate_ca_frame_times(frame_count, framerate, ca_start_time):
    return (np.arange(frame_count) + 1) / framerate + ca_start_time

def parse_cv_csv(path):
    df = pd.read_csv(path)
    potentials = df.iloc[:, 0].astype(float).values
    currents = df.iloc[:, 1].astype(float).values
    return potentials, currents

def parse_ca_csv(path):
    df = pd.read_csv(path)
    times = df[df.columns[0]].astype(float).values
    currents = df[df.columns[1]].astype(float).values
    return times, currents

# ============================================================================
# PROCESS ONE IMAGE TYPE (PNG or TIFF)
# ============================================================================
def process_image_type(image_type, bg_path, frame_files, roi_list, roi_shape_str, 
                       output_base_dir, bg_sums, roi_pixel_counts):
    """
    Process either PNG or TIFF images with the same ROIs
    """
    print(f"\n{'='*60}")
    print(f"Processing {image_type} images")
    print(f"{'='*60}")
    
    # Create output directories
    processed_dir = os.path.join(output_base_dir, "processed")
    os.makedirs(processed_dir, exist_ok=True)
    
    roi_overlay_dir = os.path.join(output_base_dir, "processed_with_ROIs")
    os.makedirs(roi_overlay_dir, exist_ok=True)

    # Process background
    bg_img = load_image(bg_path)
    if bg_img.ndim == 3:
        bg_img = extract_red_channel(bg_img)
    
    disp_bg = auto_brightness_contrast(bg_img)
    
    if image_type == "PNG":
        Image.fromarray(disp_bg).save(os.path.join(processed_dir, "processed_frame_sync.png"))
    else:
        # For TIFF, save both the display version and original
        Image.fromarray(disp_bg).save(os.path.join(processed_dir, "processed_frame_sync_display.png"))
        tifffile.imwrite(os.path.join(processed_dir, "processed_frame_sync.tiff"), bg_img)

    # Save background with ROI overlay
    overlay_bg = draw_rois_on_image(disp_bg, roi_list, roi_shape_str)
    overlay_path = os.path.join(roi_overlay_dir, f"processed_frame_sync_with_ROIs_{image_type}.png")
    Image.fromarray(overlay_bg).save(overlay_path)
    print(f"Background with ROI overlay saved: {overlay_path}")

    # Prepare storage for each ROI
    raw_intensities = [[] for _ in roi_list]
    integrated_intensities = [[] for _ in roi_list]

    # Process frames
    for idx, fname in enumerate(frame_files):
        img = load_image(fname)
        if img.ndim == 3:
            img = extract_red_channel(img)
        
        disp_img = auto_brightness_contrast(img)
        
        if image_type == "PNG":
            Image.fromarray(disp_img).save(os.path.join(processed_dir, f"processed_{idx:04d}.png"))
        else:
            # For TIFF, save both versions
            Image.fromarray(disp_img).save(os.path.join(processed_dir, f"processed_{idx:04d}_display.png"))
            tifffile.imwrite(os.path.join(processed_dir, f"processed_{idx:04d}.tiff"), img)

        # Save overlay with ROIs
        overlay_frame = draw_rois_on_image(disp_img, roi_list, roi_shape_str)
        overlay_frame_path = os.path.join(roi_overlay_dir, f"processed_{idx:04d}_ROIs_{image_type}.png")
        Image.fromarray(overlay_frame).save(overlay_frame_path)

        for i, rp in enumerate(roi_list):
            raw_sum, _ = sum_roi(img, roi_shape_str, rp)
            corrected_sum = raw_sum - bg_sums[i]
            raw_intensities[i].append(raw_sum)
            integrated_intensities[i].append(corrected_sum)

    frame_count = len(frame_files)
    print(f"Processed {frame_count} {image_type} frames")
    
    return raw_intensities, integrated_intensities, frame_count

# ============================================================================
# MAIN ANALYSIS
# ============================================================================
def analyze_and_plot():
    ca_path = None
    cv_path = None
    for ec_csv_path in csv_files:
    # Detect measurement type from filename
        basename = os.path.basename(ec_csv_path).lower()
        if 'cv' in basename:
            measurement_type = 'CV'
            cv_path = ec_csv_path
        elif 'ca' in basename:
            measurement_type = 'CA'
            ca_path = ec_csv_path
        else:
            continue

    # Load ROI frame for ROI selection
    roi_img = load_image(roi_frame_path)
    if roi_img.ndim == 2:
        disp_img = roi_img.copy()
    elif roi_img.ndim == 3 and roi_img.shape[2] >= 3:
        disp_img = cv2.cvtColor(roi_img, cv2.COLOR_RGB2BGR)
    else:
        disp_img = extract_red_channel(roi_img)

    # Select ROIs (once, will be used for both PNG and TIFF)
    print("\n" + "="*60)
    print("ROI SELECTION - These ROIs will be used for both PNG and TIFF analysis")
    print("="*60)
    roi_list = select_multiple_rois_sequential(disp_img, roi_choice, num_rois, 
                                                fixed_radius, fixed_width, fixed_height)
    if not roi_list:
        print("No ROIs selected. Exiting.")
        return
    print(f"Selected {len(roi_list)} ROIs.")

    roi_shape_str = 'circle' if roi_choice == 'c' else 'rect'

    # Create main output directories
    analysed_data_dir = os.path.join(main_folder, "Analysed_data")
    os.makedirs(analysed_data_dir, exist_ok=True)
    
    png_output_dir = os.path.join(analysed_data_dir, "png")
    tiff_output_dir = os.path.join(analysed_data_dir, "tiff")
    os.makedirs(png_output_dir, exist_ok=True)
    os.makedirs(tiff_output_dir, exist_ok=True)

    # Compute background sums from PNG background (reference for both)
    png_bg_img = extract_red_channel(load_image(png_bg_path))
    bg_sums_png = []
    roi_pixel_counts = []
    for rp in roi_list:
        s, n_pix = sum_roi(png_bg_img, roi_shape_str, rp)
        bg_sums_png.append(s)
        roi_pixel_counts.append(n_pix)
    print("PNG Background sums per ROI:", bg_sums_png)

    # Compute background sums from TIFF background
    tiff_bg_img = load_image(tiff_bg_path)
    if tiff_bg_img.ndim == 3:
        tiff_bg_img = extract_red_channel(tiff_bg_img)
    bg_sums_tiff = []
    for rp in roi_list:
        s, _ = sum_roi(tiff_bg_img, roi_shape_str, rp)
        bg_sums_tiff.append(s)
    print("TIFF Background sums per ROI:", bg_sums_tiff)

    # Process PNG images
    png_raw, png_integrated, png_frame_count = process_image_type(
        "PNG", png_bg_path, png_frame_files, roi_list, roi_shape_str, 
        png_output_dir, bg_sums_png, roi_pixel_counts
    )

    # Process TIFF images
    tiff_raw, tiff_integrated, tiff_frame_count = process_image_type(
        "TIFF", tiff_bg_path, tiff_frame_files, roi_list, roi_shape_str, 
        tiff_output_dir, bg_sums_tiff, roi_pixel_counts
    )

    # Use PNG frame count for plots (should be same as TIFF)
    frame_count = png_frame_count

    # ========================================================================
    # PLOTTING AND EXCEL GENERATION (for both PNG and TIFF)
    # ========================================================================
    if measurement_type == 'CV':
        potentials, currents = parse_cv_csv(cv_path)
        frame_potentials, frame_times = generate_frame_potentials(
            cv_start_pot, cv_end_pot, cv_scan_rate, cv_n_cycles, frame_count, framerate
        )
        current_per_cm2 = currents / electrode_area

        # Generate plots for PNG
        generate_cv_plots(potentials, currents, current_per_cm2, frame_potentials, 
                         png_integrated, roi_list, png_output_dir, "PNG")
        
        # Generate plots for TIFF
        generate_cv_plots(potentials, currents, current_per_cm2, frame_potentials, 
                         tiff_integrated, roi_list, tiff_output_dir, "TIFF")

        # Excel for PNG
        save_cv_excel(potentials, currents, current_per_cm2, frame_times, frame_potentials,
                     png_raw, png_integrated, roi_list, electrode_area, frame_count,
                     png_output_dir, "PNG")
        
        # Excel for TIFF
        save_cv_excel(potentials, currents, current_per_cm2, frame_times, frame_potentials,
                     tiff_raw, tiff_integrated, roi_list, electrode_area, frame_count,
                     tiff_output_dir, "TIFF")

    elif measurement_type == 'CA':
        ca_times, currents = parse_ca_csv(ca_path)
        frame_times = generate_ca_frame_times(frame_count, framerate, ca_start_time)
        current_per_cm2 = currents / electrode_area

        # Generate plots for PNG
        generate_ca_plots(ca_times, currents, current_per_cm2, frame_times,
                         png_integrated, roi_list, png_output_dir, "PNG")
        
        # Generate plots for TIFF
        generate_ca_plots(ca_times, currents, current_per_cm2, frame_times,
                         tiff_integrated, roi_list, tiff_output_dir, "TIFF")

        # Excel for PNG
        save_ca_excel(ca_times, currents, current_per_cm2, frame_times,
                     png_raw, png_integrated, roi_list, electrode_area, frame_count,
                     ca_applied_potential, png_output_dir, "PNG")
        
        # Excel for TIFF
        save_ca_excel(ca_times, currents, current_per_cm2, frame_times,
                     tiff_raw, tiff_integrated, roi_list, electrode_area, frame_count,
                     ca_applied_potential, tiff_output_dir, "TIFF")

    print("\n" + "=" * 60)
    print("ANALYSIS COMPLETE")
    print(f"PNG results saved in: {png_output_dir}")
    print(f"TIFF results saved in: {tiff_output_dir}")
    print("=" * 60)

def generate_cv_plots(potentials, currents, current_per_cm2, frame_potentials, 
                     integrated_intensities, roi_list, output_dir, image_type):
    # Custom colors for multiple ROIs - use crimson variations
    ecl_colors = ['crimson', 'darkred', 'indianred', 'lightcoral', 'salmon', 
                  'tomato', 'orangered', 'coral', 'darkorange', 'orange']
    
    # Plot 1: Measured current + ECL
    fig, ax1 = plt.subplots(figsize=(10, 6))
    ax1.plot(potentials, currents * 1000, color='dodgerblue', linewidth=3, 
             label='Measured Current (mA)')
    ax1.set_xlabel('Potential (V)', fontsize=24, fontweight='bold')
    ax1.set_ylabel('Current (mA)', color='dodgerblue', fontsize=24, fontweight='bold')
    ax1.tick_params(axis='y', labelcolor='dodgerblue', width=3, length=10, labelsize=18)
    ax1.tick_params(axis='x', width=3, length=10, labelsize=18)

    ax2 = ax1.twinx()
    for i in range(len(roi_list)):
        color = ecl_colors[i % len(ecl_colors)]
        ax2.plot(frame_potentials, integrated_intensities[i],
                 color=color, marker='o', markersize=10,
                 markerfacecolor='lightcoral', markeredgewidth=1.5,
                 linewidth=3, alpha=0.9, label=f'ROI{i + 1} ECL (Raw-bg)')
    ax2.set_ylabel('ECL Intensity (counts)', color='crimson', fontsize=24, fontweight='bold')
    ax2.tick_params(axis='y', labelcolor='crimson', width=3, length=10, labelsize=18)
    ax2.tick_params(axis='x', width=3, length=10, labelsize=18)

    lines_1, labels_1 = ax1.get_legend_handles_labels()
    lines_2, labels_2 = ax2.get_legend_handles_labels()
    ax1.legend(lines_1 + lines_2, labels_1 + labels_2, fontsize=18, loc='best', frameon=False)

    plt.tight_layout()
    plt.savefig(os.path.join(output_dir, f'cv_overlay_multiROI_{image_type}.png'), dpi=300)
    plt.close()

    # Plot 2: Normalized (current density vs ECL)
    fig, ax1 = plt.subplots(figsize=(10, 6))
    ax1.plot(potentials, current_per_cm2 * 1000, color='dodgerblue', linewidth=3, 
             label='Current density')
    ax1.set_xlabel('Potential (V)', fontsize=24, fontweight='bold')
    ax1.set_ylabel('Current density (mA/cm²)', color='dodgerblue', fontsize=24, fontweight='bold')
    ax1.tick_params(axis='y', labelcolor='dodgerblue', width=3, length=10, labelsize=18)
    ax1.tick_params(axis='x', width=3, length=10, labelsize=18)

    ax2 = ax1.twinx()
    for i in range(len(roi_list)):
        color = ecl_colors[i % len(ecl_colors)]
        ax2.plot(frame_potentials, integrated_intensities[i],
                 color=color, marker='o', markersize=10,
                 markerfacecolor='lightcoral', markeredgewidth=1.5,
                 linewidth=3, alpha=0.9, label=f'ROI{i + 1} ECL/cm² (Raw-bg)')
    ax2.set_ylabel('ECL Intensity (counts)', color='crimson', fontsize=24, fontweight='bold')
    ax2.tick_params(axis='y', labelcolor='crimson', width=3, length=10, labelsize=18)
    ax2.tick_params(axis='x', width=3, length=10, labelsize=18)

    lines_1, labels_1 = ax1.get_legend_handles_labels()
    lines_2, labels_2 = ax2.get_legend_handles_labels()
    ax1.legend(lines_1 + lines_2, labels_1 + labels_2, fontsize=18, loc='best', frameon=False)

    plt.tight_layout()
    plt.savefig(os.path.join(output_dir, f'cv_overlay_percm2_multiROI_{image_type}.png'), dpi=300)
    plt.close()

def save_cv_excel(potentials, currents, current_per_cm2, frame_times, frame_potentials,
                 raw_intensities, integrated_intensities, roi_list, electrode_area, 
                 frame_count, output_dir, image_type):
    rows = []
    max_len = max(len(potentials), frame_count)
    for i in range(max_len):
        pot_val = potentials[i] if i < len(potentials) else ''
        curr_val = currents[i] if i < len(currents) else ''
        curr_percm2_val = curr_val / electrode_area if i < len(currents) else ''
        frame_idx = i if i < frame_count else ''
        frame_time_val = frame_times[i] if i < len(frame_times) else ''
        frame_pot_val = frame_potentials[i] if i < len(frame_potentials) else ''

        roi_raw_vals = []
        roi_bg_vals = []
        roi_bg_percm2_vals = []
        for r in range(len(roi_list)):
            roi_raw = raw_intensities[r][i] if i < len(raw_intensities[r]) else ''
            roi_bg = integrated_intensities[r][i] if i < len(integrated_intensities[r]) else ''
            roi_bg_percm2 = roi_bg / electrode_area if roi_bg != '' else ''
            roi_raw_vals.append(roi_raw)
            roi_bg_vals.append(roi_bg)
            roi_bg_percm2_vals.append(roi_bg_percm2)

        row = [pot_val, curr_val, curr_percm2_val, frame_idx, frame_time_val, frame_pot_val]
        for r in range(len(roi_list)):
            row.extend([roi_raw_vals[r], roi_bg_vals[r], roi_bg_percm2_vals[r]])
        rows.append(row)

    header = ['Measured Potential (V vs Ag)', 'Measured Current (A)', 'Current density (A/cm²)',
              'Frame', 'Frame Time (s)', 'Frame Potential (V vs Ag)']
    for r in range(len(roi_list)):
        header.extend([f'ROI{r + 1} ECL Raw', f'ROI{r + 1} ECL Raw-bg', f'ROI{r + 1} ECL/cm² Raw-bg'])

    df = pd.DataFrame(rows, columns=header)
    excel_path = os.path.join(output_dir, f'cv_multiROI_data_{image_type}.xlsx')
    df.to_excel(excel_path, index=False)
    print(f"CV {image_type} Excel saved: {excel_path}")

def generate_ca_plots(ca_times, currents, current_per_cm2, frame_times,
                     integrated_intensities, roi_list, output_dir, image_type):
    # Custom colors for multiple ROIs - use crimson variations
    ecl_colors = ['crimson', 'darkred', 'indianred', 'lightcoral', 'salmon', 
                  'tomato', 'orangered', 'coral', 'darkorange', 'orange']
    
    # Plot 1: Measured current vs time + ECL
    fig, ax1 = plt.subplots(figsize=(10, 6))
    ax1.plot(ca_times, currents * 1000, color='dodgerblue', linewidth=3, 
             label='Measured Current (mA)')
    ax1.set_xlabel('Time (s)', fontsize=24, fontweight='bold')
    ax1.set_ylabel('Current (mA)', color='dodgerblue', fontsize=24, fontweight='bold')
    ax1.tick_params(axis='y', labelcolor='dodgerblue', width=3, length=10, labelsize=18)
    ax1.tick_params(axis='x', width=3, length=10, labelsize=18)

    ax2 = ax1.twinx()
    for i in range(len(roi_list)):
        color = ecl_colors[i % len(ecl_colors)]
        ax2.plot(frame_times, integrated_intensities[i],
                 color=color, marker='o', markersize=10,
                 markerfacecolor='lightcoral', markeredgewidth=1.5,
                 linewidth=3, alpha=0.9, label=f'ROI{i + 1} ECL (bg-sub)')
    ax2.set_ylabel('ECL Intensity (counts)', color='crimson', fontsize=24, fontweight='bold')
    ax2.tick_params(axis='y', labelcolor='crimson', width=3, length=10, labelsize=18)
    ax2.tick_params(axis='x', width=3, length=10, labelsize=18)

    lines_1, labels_1 = ax1.get_legend_handles_labels()
    lines_2, labels_2 = ax2.get_legend_handles_labels()
    ax1.legend(lines_1 + lines_2, labels_1 + labels_2, fontsize=18, loc='best', frameon=False)

    plt.tight_layout()
    plt.savefig(os.path.join(output_dir, f'ca_overlay_multiROI_{image_type}.png'), dpi=300)
    plt.close()

    # Plot 2: Normalized (current density)
    fig, ax1 = plt.subplots(figsize=(10, 6))
    ax1.plot(ca_times, current_per_cm2 * 1000, color='dodgerblue', linewidth=3, 
             label='Current density')
    ax1.set_xlabel('Time (s)', fontsize=24, fontweight='bold')
    ax1.set_ylabel('Current density (mA/cm²)', color='dodgerblue', fontsize=24, fontweight='bold')
    ax1.tick_params(axis='y', labelcolor='dodgerblue', width=3, length=10, labelsize=18)
    ax1.tick_params(axis='x', width=3, length=10, labelsize=18)

    ax2 = ax1.twinx()
    for i in range(len(roi_list)):
        color = ecl_colors[i % len(ecl_colors)]
        ax2.plot(frame_times, integrated_intensities[i],
                 color=color, marker='o', markersize=10,
                 markerfacecolor='lightcoral', markeredgewidth=1.5,
                 linewidth=3, alpha=0.9, label=f'ROI{i + 1} ECL')
    ax2.set_ylabel('ECL Intensity (counts)', color='crimson', fontsize=24, fontweight='bold')
    ax2.tick_params(axis='y', labelcolor='crimson', width=3, length=10, labelsize=18)
    ax2.tick_params(axis='x', width=3, length=10, labelsize=18)

    lines_1, labels_1 = ax1.get_legend_handles_labels()
    lines_2, labels_2 = ax2.get_legend_handles_labels()
    ax1.legend(lines_1 + lines_2, labels_1 + labels_2, fontsize=18, loc='best', frameon=False)

    plt.tight_layout()
    plt.savefig(os.path.join(output_dir, f'ca_overlay_percm2_multiROI_{image_type}.png'), dpi=300)
    plt.close()

def save_ca_excel(ca_times, currents, current_per_cm2, frame_times,
                 raw_intensities, integrated_intensities, roi_list, electrode_area, 
                 frame_count, ca_applied_potential, output_dir, image_type):
    rows = []
    max_len = max(len(ca_times), len(frame_times))
    for i in range(max_len):
        meas_time = ca_times[i] if i < len(ca_times) else ''
        meas_curr = currents[i] if i < len(currents) else ''
        meas_curr_percm = meas_curr / electrode_area if i < len(currents) else ''
        frame_idx = i if i < len(frame_times) else ''
        frame_time = frame_times[i] if i < len(frame_times) else ''

        roi_raw_vals = []
        roi_bg_vals = []
        roi_bg_percm2_vals = []
        for r in range(len(roi_list)):
            roi_raw = raw_intensities[r][i] if i < len(raw_intensities[r]) else ''
            roi_bg = integrated_intensities[r][i] if i < len(integrated_intensities[r]) else ''
            roi_bg_percm2 = roi_bg / electrode_area if roi_bg != '' else ''
            roi_raw_vals.append(roi_raw)
            roi_bg_vals.append(roi_bg)
            roi_bg_percm2_vals.append(roi_bg_percm2)

        # Find closest measured current to this frame time
        if i < len(frame_times) and len(ca_times) > 0:
            idx_closest = np.argmin(np.abs(ca_times - frame_times[i]))
            closest_curr = currents[idx_closest]
            closest_curr_percm = closest_curr / electrode_area
        else:
            closest_curr = ''
            closest_curr_percm = ''

        row = [meas_time, meas_curr, meas_curr_percm, frame_idx, frame_time]
        for r in range(len(roi_list)):
            row.extend([roi_raw_vals[r], roi_bg_vals[r], roi_bg_percm2_vals[r]])
        row.extend([ca_applied_potential, closest_curr, closest_curr_percm])
        rows.append(row)

    header = ['Measured Time (s)', 'Measured Current (A)', 'Measured Current density (A/cm2)',
              'Frame Index', 'Frame Time (s)']
    for r in range(len(roi_list)):
        header.extend([f'ROI{r + 1} ECL Raw', f'ROI{r + 1} ECL Raw-bg', f'ROI{r + 1} ECL/cm² Raw-bg'])
    header.extend(['Applied Potential (V vs Ag)', 'Closest Measured Current to Frame (A)',
                   'Closest Current /cm2 to Frame (A/cm2)'])

    df = pd.DataFrame(rows, columns=header)
    excel_path = os.path.join(output_dir, f'ca_multiROI_data_{image_type}.xlsx')
    df.to_excel(excel_path, index=False)
    print(f"CA {image_type} Excel saved: {excel_path}")

if __name__ == "__main__":
    analyze_and_plot()